if tool:GetID() ~= "Stroke" then
	print("This script only works with Strokes")
	return
end


function AddParam(tkey, val)
	tkey = val
	tkey.__ctor = "Input"
end

-- untransform all points and add them
function GetPoints(dest, src, starttime, center, angle, size, aspect)
	local i, pt
	angle = angle * math.pi / 180
	local sa = math.sin(angle)
	local ca = math.cos(angle)
	
	for i, pt in ipairs(src) do
		local x =  ca * pt.X - sa * (pt.Y / aspect)
		local y = (sa * pt.X + ca * (pt.Y / aspect)) * aspect
		
		dest[i] = {}
		dest[i].X = x * size + center[1] - 0.5
		dest[i].Y = y * size + center[2] - 0.5
		dest[i].T = pt.Time - starttime
		dest[i].Pressure = pt.Pressure
	end
end

-- Combines each stroke's inputs into an ongoing table of params
function MergeStrokeParams(dest, src)
	local key, val
	for key, val in pairs(src) do
		if type(val) == "table" then
			if type(dest[key]) ~= "table" then
				dest[key] = {}
			end
			MergeStrokeParams(dest[key], val)
		else
			dest[key] = val
		end
	end
end

-- Checks for params missing from the stroke's saved values, and fetches defaults
function FillInDefaults(stroke, params, values)
	local pname, p, val
	for pname, p in pairs(params) do
		if values[pname] == nil then
			local inp = stroke[pname]
			if inp then
				local attrs = inp:GetAttrs()

				if attrs.INPS_DataType == "Number" then
					val = attrs.INPN_Default
				elseif attrs.INPS_DataType == "Point" then
					val = { attrs.INPN_DefaultX, attrs.INPN_DefaultY }
				elseif attrs.INPS_DataType == "Text" then
					val = attrs.INPN_DefaultText
				end

				params[pname] = { Value = val }
			end
		end
	end
end

function CopyTable(tbl)
	local i,v,ret
	ret = {}  
	for i,v in pairs(tbl) do
		if type(v) == "table" then  
			ret[i] = CopyTable(v)  
		else  
			ret[i] = v  
		end  
	end  
	return ret  
end  



strokenames = {}
local first = tool
local last = tool
local t = tool

-- find first selected tool
while t do
	local attrs = t:GetAttrs()
	if attrs.TOOLB_Selected and t:GetID() == "Stroke" then
		first = t
	end
	
	if t.Paint then
		local out = t.Paint:GetConnectedOutput()
		if out then
			t = out:GetTool()
		else
			t = nil
		end
	else
		t = nil
	end
end

-- find last selected tool
t = first
while t do
	local attrs = t:GetAttrs()
	if attrs.TOOLB_Selected and t:GetID() == "Stroke" then
		last = t
		table.insert(strokenames, attrs.TOOLS_Name)
	end
	
	local out = t:FindMainOutput(1)
	if out then
		local inp = out:GetConnectedInputs()[1]
		if inp then
			t = inp:GetTool()
		else
			t = nil
		end
	else
		t = nil
	end
end

if #strokenames == 0 then
	print("Error - you must select some strokes first.")
	return
end

local i,j, name, pname,p, inp
local allsettings = last:SaveSettings()	-- saves all strokes above it too
local ms =
{
	__ctor = "Multistroke",
	IsThreaded = true,
	Strokes = { }
}

-- Get some default params
local params = {}


-- process stroke list
for i,name in ipairs(strokenames) do
	print("Converting "..name)

	local settings = allsettings.Tools[name]
	local stroke = {}

	-- get params
	if settings.Inputs then
		for pname,p in pairs(settings.Inputs) do
			if pname ~= "Paint" then
				if type(params[pname]) ~= "table" then
					params[pname] = {}
				end					
					
				MergeStrokeParams(params[pname], p)
				params[pname].__ctor = "Input"
			end
		end
	end
	
	-- get any necessary default values
	FillInDefaults(comp[name], params, settings.Inputs)

	-- get stroke timing
	if settings.EnabledRegion then
		stroke.Time   = settings.EnabledRegion[1].Start
		stroke.Frames = settings.EnabledRegion[1].End + 0.0001 - settings.EnabledRegion[1].Start
		AddParam(params.Duration, { Value = stroke.Frames })
	else
		stroke.Time   = -1000000		-- all time
		stroke.Frames =  2000000
	end

	-- get stroke transform settings
	local center = { 0.5, 0.5 }
	local angle = 0
	local size = 1
	local attrs = comp[name]:GetAttrs()
	local aspect = 1

	if params.Center then
		center = params.Center.Value
	end
	if params.StrokeAngle then
		angle = params.StrokeAngle.Value
	end
	if params.StrokeSize then
		size = params.StrokeSize.Value
	end
	if attrs.TOOLI_ImageWidth ~= nil and attrs.TOOLI_ImageWidth > 0 and attrs.TOOLI_ImageHeight > 0 then
		aspect = attrs.TOOLI_ImageWidth / attrs.TOOLI_ImageHeight * attrs.TOOLN_ImageAspectX / attrs.TOOLN_ImageAspectY
	else
		print("Warning - Aspect may be incorrect. View or render the Paint tool first, before combining strokes.")
	end
	
	-- get untransformed stroke points
	if type(settings.Points) == "table" then
		local starttime = settings.Points[1].Time
		GetPoints(stroke, settings.Points, starttime, center, angle, size, aspect)
	end
	
	-- kill any transformations
	params.Center = nil
	params.StrokeAngle = nil
	params.StrokeSize = nil
	-- have to adjust brush size too
	if size ~= 1 and params.BrushShape then
		local brushsize = params.BrushShape.Value[1]..".Size"
		if params[brushsize] then
			params[brushsize].Value = params[brushsize].Value * size
		end
	end

	-- look for image connections (TODO: other datatypes too?)
	local inplist = comp[name]:GetInputList("Image")
	for j,inp in pairs(inplist) do
		local out = inp:GetConnectedOutput()
		
		if out then
			local id = inp.ID

			if id == nil then                  -- for older versions
				local inpattrs = inp:GetAttrs()
				if inpattrs.INPS_DataType == "Image" then
					id = inpattrs.INPS_ID
				end
			end

			if id ~= "Paint" then
				if params[id] == nil then
					params[id] = {}
				end

				-- save Time
				local t = stroke.Time
				if id == "PaintApplyRubThrough.SourceTool" and params["PaintApplyRubThrough.TimeOffset"] then
					t = t + params["PaintApplyRubThrough.TimeOffset"].Value
				end
				params[id].Time = t
			end
		end
	end

	-- add the stroke
	stroke.Params = CopyTable(params)
	table.insert(ms.Strokes, stroke)
end	


comp:StartUndo("Combine strokes")

comp:Lock()

local msname = "Strokes"
i = 1
while comp[msname..i] do
	i = i + 1
end
msname = msname..i


-- Add it to the comp
comp:Paste( { Tools = { [msname] = ms } } )

-- Delete all the other strokes (not 'tool')
for i, name in ipairs(strokenames) do
	local t = comp[name]
	if t and t ~= tool then
		t:Delete()
	end
end

-- and connect it up where 'tool' is
local mstool = comp[msname]
local srcout = tool.Paint:GetConnectedOutput()
if srcout and mstool then
	mstool.Paint:ConnectTo(srcout)
end

local inps = tool.Out:GetConnectedInputs()
if inps[1] then
	inps[1]:ConnectTo(mstool.Out)		-- should delete 'tool'
end

comp:Unlock()

comp:EndUndo(true)

print(#strokenames.." strokes converted.")
